// Version: 2025_02_27
// Compiliert mit IDE 2.3.2

#include <util/crc16.h>
#include <avr/pgmspace.h>
#include <Wire.h>
#include <EEPROM.h>
#include <LiquidCrystal.h>

//=============================================================================
// Konstanten
//=============================================================================
/* 
  Belegung: 34 poliges Flachbandkabel / Pfostestecker (33 = Sonde und 34 = Masse)
    1   3   5   7   9  11  13  15  17  19  21  23  25  27  29  31 (33)
      2   4   6   8  10  12  14  16  18  20  22  24  26  28  30  32 (34)

  Belegung: 37 polige Sub-D-Buchse (17 = Sonde und 36 = Masse; 18, 19 und 37 unbenutzt)
    1   2   3   4   5   6   7   8   9  10  11  12  13  14  15  16 (17  18  19) 
      20  21  22  23  24  25  26  27 28  29  30  31  32  33  34  35 (36  37)

  Belegung: Adapter
  ---------------A-----------------|--------------B-----------------
    1   3   5   7   9  11  13  15   1   3   5   7   9  11  13  15  
      2   4   6   8  10  12  14  16   2   4   6   8  10  12  14  16 
*/

/* 
  Anzeigemodi:
  A: Angaben beziehen sich auf Sub-D-Stecker (oben: 1-16, unten: 20-35, 17-19 und 36-37 keine Pins)
  B: Angaben beziehen sich auf 34 poligen Pfostenstecker (33 und 34 keine Pins)
  C: Angaben beziehen sich auf Adpater, Sortierreihenfolge: A1, A2, ..., A16, B1, B2, ..., B16
  D: Angaben beziehen sich auf Adpater, Sortierreihenfolge: 1A, 1B, 2A, 2B, ..., 16A, 16B
*/
const int ANZAHL_ANZEIGE_MODUS=4;

// Legt die Namen der Pins fest
const char PinNamen0[] PROGMEM = { " S1" " S2" " S3" " S4" " S5" " S6" " S7" " S8" " S9" "S10" "S11" "S12" "S13" "S14" "S15" "S16" "S20" "S21" "S22" "S23" "S24" "S25" "S26" "S27" "S28" "S29" "S30" "S31" "S32" "S33" "S34" "S35" }; 
const char PinNamen1[] PROGMEM = { " P1" " P2" " P3" " P4" " P5" " P6" " P7" " P8" " P9" "P10" "P11" "P12" "P13" "P14" "P15" "P16" "P17" "P18" "P19" "P20" "P21" "P22" "P23" "P24" "P25" "P26" "P27" "P28" "P29" "P30" "P31" "P32" };
const char PinNamen2[] PROGMEM = { " A1" " A2" " A3" " A4" " A5" " A6" " A7" " A8" " A9" "A10" "A11" "A12" "A13" "A14" "A15" "A16" " B1" " B2" " B3" " B4" " B5" " B6" " B7" " B8" " B9" "B10" "B11" "B12" "B13" "B14" "B15" "B16" };
const char PinNamen3[] PROGMEM = { " 1A" " 1B" " 2A" " 2B" " 3A" " 3B" " 4A" " 4B" " 5A" " 5B" " 6A" " 6B" " 7A" " 7B" " 8A" " 8B" " 9A" " 9B" "10A" "10B" "11A" "11B" "12A" "12B" "13A" "13B" "14A" "14B" "15A" "15B" "16A" "16B" };

const char* const PinNamen[] PROGMEM = { PinNamen0, PinNamen1, PinNamen2, PinNamen3 }; // String-Array

// Legt die Reigenfolge fest, in der die Pins ausgewertet werden
const byte PinFolge0[] PROGMEM = {  0,  2,  4,  6,  8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30,  1,  3,  5,  7,  9, 11, 13, 15, 17, 19, 21, 23, 25, 27, 29, 31 };
const byte PinFolge1[] PROGMEM = {  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31 };
const byte PinFolge2[] PROGMEM = {  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31 };
const byte PinFolge3[] PROGMEM = {  0, 16,  1, 17,  2, 18,  3, 19,  4, 20,  5, 21,  6, 22,  7, 23,  8, 24,  9, 25, 10, 26, 11, 27, 12, 28, 13, 29, 14, 30, 15, 31 };

const byte* const PinFolgen[] PROGMEM = {PinFolge0, PinFolge1, PinFolge2, PinFolge3};

/* Prüfcodes für n Verbindungen 1:1 (A1-B1, A2-B2, A3-B3, ...)
n: 0    1    2    3    4    5    6    7    8    9   10   11   12   13   14   15   16
 0000 9E2C 7E39 594B 5E31 4859 B698 55A9 79F2 2B56 2AF4 9AC5 AAAF 65FC 59B7 1122 8427
*/

const char Titel1_Text[] PROGMEM = "****************";
const char Titel2_Text[] PROGMEM = "* Verbindungs- *";
const char Titel3_Text[] PROGMEM = "* pruefer F2025*"; 
const char* const Titel_Texte[4] PROGMEM = {Titel1_Text, Titel2_Text, Titel3_Text, Titel1_Text};

const char Lcd_Kontrast_Text[] PROGMEM  = "LCD-Kontrast: ";
const char Verbindungen_Text[] PROGMEM  = "  Verbindungen  ";
const char Gespeichert_Text[] PROGMEM   = "  gespeichert   ";
const char Sonde_hat_Text[] PROGMEM     = "   Sonde hat    ";
const char Masseschluss_Text[] PROGMEM  = "  Masseschluss  ";
const char Masseschluss2_Text[] PROGMEM = "Masseschluss:";
const char Sonde_Text[] PROGMEM         = "Sonde:.. Verb.";
const char Verb_Text[] PROGMEM          = ": .. Verb.";
const char Fehler_Text[] PROGMEM        = ":.. Fehler";
const char Pin_Fehlt_Text[] PROGMEM     = "Pin fehlt: ";
const char Pin_Zuviel_Text[] PROGMEM    = "Pin zuviel: ";
const char Fehlt_Text[] PROGMEM         = " fehlt";
const char Zuviel_Text[] PROGMEM        = " zuviel";
const char Getauscht_Text[] PROGMEM     = " vertaus.";
const char Oder_Text[] PROGMEM          = " oder ";
const char I2C_Fehler_Text[] PROGMEM    = "I2C-Fehler:";
const char Pin_Fehler_Text[] PROGMEM    = "Pin-Fehler:";
const char Und_weitere_Text[] PROGMEM   = "und weitere";
const char Trennlinie_Text[] PROGMEM    = "----------------";
const char PCF8574_Text[] PROGMEM       = "PCF8574";
const char PCF8574A_Text[] PROGMEM      = "PCF8574A";
const char PCF8574fehlt_Text[] PROGMEM  = "PCF8574 fehlt";

const int UMIN=200;
const int UMAX=800;

const int TASTE_NULL=0; // | Scroll | Mode | LCD | Mode        |
const int TASTE_VCC=1;  // |   Ab   |   Z  |  -  | Speichern   |
const int TASTE_GND=2;  // |   Auf  |   A  |  +  | Vergleichen |

const byte CODE_PFEIL_AUF= '\0';
const byte CODE_PFEIL_AB = '\1';
// const byte CODE_PFEIL_AUF='$';
// const byte CODE_PFEIL_AB ='%';

const char ANZEIGE_MODUS_TRENNER='>';
const char VERBINDUNG_TRENNER='-'; // ~ ist beim LCD ein Pfeil nach rechts ->
const char PIN_ZUVIEL_TRENNER='-';
const char PIN_FEHLT_TRENNER='-';
const char PIN_GETAUSCHT_TRENNER='&';
const char VERBINDUNGS_ABSCHLUSS=' ';
const char LOESCH_ZEICHEN=' ';
const char FUELL_ZEICHEN=' ';
const char LEER_ZEICHEN=' ';
const char NULL_ZEICHEN='\0';
const int PIN_MAMEN_LAENGE=3;

const int ANZAHL_PINS=32;
const int ANZAHL_BYTES=ANZAHL_PINS/8;
const int ANZAHL_KNOTEN_FEHLT_ZUVIEL=2;
    
const int LCD_ZEILEN=4;
const int LCD_SPALTEN=16;
const int MAX_BUF_ZEILEN=21;
const int BUF_SPALTEN=LCD_SPALTEN;
const int BUF_ZEICHEN=(BUF_SPALTEN*MAX_BUF_ZEILEN);

const int ANZAHL_VERBIN_POS=8;
const int ANZAHL_FEHLER_POS=7;

const int MAX_ZAHL=99;

const int ANZEIGE_TON_DAUER=100;
const int SPEICHERN_TON_DAUER=300;
const int TITEL_ANZEIGE_DAUER=2000; 
const int ZEIGE_LCD_KONTRAST_DAUER=10;
const int ZEIGE_GESPEICHERT_DAUER=10;
const int LOOP_DAUER=250;
const int ABTAST_DELAY=1;

const byte MIN_LCD_KONTRAST=1;
const byte MAX_LCD_KONTRAST=10;

const int MASSESCHLUSS_FEHLER=-1;
const int PIN_FEHLER=-2;

const word CRC_START=0x7CC7;

const int ABTASTUNGEN_EEPROM_ADRESSE=0;
const int GELERNT_CRC_EEPROM_ADRESSE=ABTASTUNGEN_EEPROM_ADRESSE+ANZAHL_PINS*ANZAHL_BYTES;
const int ANZEIGEMODUS_EEPROM_ADRESSE=GELERNT_CRC_EEPROM_ADRESSE+2;
const int LCD_KONTRAST_EEPROM_ADRESSE=ANZEIGEMODUS_EEPROM_ADRESSE+2;

const int PCF8574_SLAVE_ADR =B0100000; // Slave-Adresse PCF8574
const int PCF8574A_SLAVE_ADR=B0111000; // Slave-Adresse PCF8574A

const int BETRIEBSMODUS_EINGANG_PIN=4;
const int SONDE_EINGANG_PIN=5;
const int LCD_KONTRAST_AUSGANG_PIN=6;
const int PIEZO_AUSGANG_PIN=7;

const int LCD_KONTRAST_SPANNUNG_PIN=A0;
const int SCROLLEN_SPANNUNG_PIN=A1;
const int ANZEIGE_MODUS_SPANNUNG_PIN=A2;
const int SPEICHERN_VERGLEICHEN_SPANNUNG_PIN=A3;

// Beachten: A4=SDA, A5=SCL

//=============================================================================
// Variablen
//=============================================================================

union Abtastung{
  byte B[4];        // für Byte-weisen Zugriff
  unsigned long UL; // für 32-Bit zugriff
};

union Abtastung IstAbtastungen[ANZAHL_PINS];
union Abtastung SollAbtastungen[ANZAHL_PINS];
union Abtastung FehlAbtastungen[ANZAHL_PINS];

unsigned long SondeVerbindungen;

word AbtastungenCrc;
word ZeichenBufferCrc, AltZeichenBufferCrc;
word ZeigeLcdKontrastTimer;
word ZeigeGespeichertTimer;
char ZeichenBuffer[BUF_ZEICHEN];
int BufferZeile, BufferSpalte;
int ZeilenOffset, MaxZeilenOffset;

word PinFolgenOffset;
word PinNamenOffset;

int AnzahlFehler;
unsigned long PinFehlerPins;
int AnzahlPinFehler;
boolean VerbindungenDa;

boolean Buffervoll;
boolean Trennung;

byte AnzeigeModus;
int AnzeigeModusTaste;
boolean DirektAnzeigen;

byte LcdKontrast;
int LcdKontrastTaste;

boolean BetriebsModus; // L=Sonde, H=Verbindungen
boolean SpeichernModus;
boolean VergleichenModus;

byte PfeilAuf[8] = { 
  B00100, 
  B01110, 
  B10101, 
  B00100, 
  B00100, 
  B00100, 
  B11111, 
  B00000 };

byte PfeilAb[8]  = { 
  B11111, 
  B00100, 
  B00100, 
  B00100, 
  B10101, 
  B01110, 
  B00100, 
  B00000 };

int PCF8574Adr[ANZAHL_BYTES]={-1, -1, -1, -1};

//=============================================================================

LiquidCrystal Lcd(8, 9, 10, 11, 12, 13); // E, RS, D4, D5, D6, D7

//=============================================================================

void piezoPuls(const unsigned long PulsDauer) {
  digitalWrite(PIEZO_AUSGANG_PIN, HIGH);
  delay(PulsDauer);
  digitalWrite(PIEZO_AUSGANG_PIN, LOW);
}
//=============================================================================

int zaehlenEinsen(unsigned long Daten) {
  int Zaehler=0;
  while (Daten!=0) { Zaehler++; Daten&=(Daten-1); }
  return Zaehler;
}
//=============================================================================

int suchenEins(const unsigned long Daten, int Pos) {
  while (!bitRead(Daten, Pos) && Pos<=32) Pos++;
  return Pos;
}
//=============================================================================

word rechnenCrc(const byte *Daten, const unsigned int Laenge) {
  word Crc=CRC_START;
  for (unsigned int l=0; l<Laenge; l++)
    Crc=_crc_xmodem_update(Crc, *Daten++);
  if (Crc==0 && VerbindungenDa) Crc=CRC_START;
  return Crc;
}
//=============================================================================
//=============================================================================

void loeschenBufferZeile(const int Zeile) {
  int LoeschPosi=Zeile*BUF_SPALTEN;
  for (int Spalte=0; Spalte<BUF_SPALTEN; Spalte++)
    ZeichenBuffer[LoeschPosi+Spalte]=LOESCH_ZEICHEN;
}
//=============================================================================

void loeschenBufferZeilen() {
  for (int Zeile=0; Zeile<MAX_BUF_ZEILEN; Zeile++) loeschenBufferZeile(Zeile);
  BufferZeile=0; BufferSpalte=0; 
  Buffervoll=false;
  Trennung=false;
}
//=============================================================================

void einfuegenTrennlinie() {
  Trennung=false;
  einfuegenPGMText(Trennlinie_Text);
}
//=============================================================================

void einfuegenZeichen(const byte Zeichen) {
  if (Buffervoll) return;
  if (Trennung) einfuegenTrennlinie();
  if (Buffervoll) return; // durch Trennlinie
  ZeichenBuffer[BufferZeile*BUF_SPALTEN+BufferSpalte]=Zeichen;
  BufferSpalte++;
  if (BufferSpalte>=BUF_SPALTEN) {
    BufferSpalte=0;
    if (BufferZeile<MAX_BUF_ZEILEN-1) {
      BufferZeile++; 
    } else { 
      loeschenBufferZeile(BufferZeile); 
      einfuegenPGMText(Und_weitere_Text); 
      Buffervoll=true; 
    }
  }
  MaxZeilenOffset=BufferZeile-(LCD_ZEILEN-1);
  if (BufferSpalte==0) MaxZeilenOffset--;
  if (MaxZeilenOffset<0) MaxZeilenOffset=0; 
}
//=============================================================================

void einfuegenNeueZeile()  {
  while (!Buffervoll && BufferSpalte!=0) einfuegenZeichen(FUELL_ZEICHEN);
}
//=============================================================================

void einfuegenHexNibble(byte Nibble) {
  Nibble&=0x0F;
  Nibble=Nibble<10 ? Nibble+'0' : Nibble+'A'-10;
  einfuegenZeichen(Nibble);
}
//=============================================================================

void einfuegenHexByte(const byte Byt) {
  einfuegenHexNibble(Byt>>4);
  einfuegenHexNibble(Byt);
}
//=============================================================================

void einfuegenHexWord(const word Wrd) {
  einfuegenHexByte(Wrd>>8);
  einfuegenHexByte(Wrd);
}
//=============================================================================

void einfuegenText(const char *Text) {
  while (!Buffervoll && *Text!=NULL_ZEICHEN) {
    einfuegenZeichen(*Text);
    Text++;
  }
}
//=============================================================================

void einfuegenPGMText(const char *Text) {
  char Zeichen=pgm_read_byte(Text);
  while (!Buffervoll && Zeichen!=NULL_ZEICHEN) {
    einfuegenZeichen(Zeichen); 
    Text++;
    Zeichen=pgm_read_byte(Text);
   }
}
//=============================================================================

void einfuegenZahl(int Zahl, const char Trenner) {
  char Buffer[8];
  if (Zahl>MAX_ZAHL) Zahl=MAX_ZAHL;
  itoa(Zahl, Buffer, 10); // 10=Dezimalsystem
  einfuegenText(Buffer);
  if (Trenner!=NULL_ZEICHEN) einfuegenZeichen(Trenner);
}
//=============================================================================

void nachtragenZahlPosi(int Zahl, const int Posi) {
  if (Zahl>MAX_ZAHL) Zahl=MAX_ZAHL;
  int Zehner=Zahl/10; int Einer=Zahl%10;
  if (Zehner!=0) ZeichenBuffer[Posi]='0'+Zehner; else ZeichenBuffer[Posi]=LEER_ZEICHEN;
  ZeichenBuffer[Posi+1]='0'+Einer;
}
//=============================================================================
//=============================================================================

void ausgebenLcdZeile(int Zeile) {
  Zeile=Zeile+ZeilenOffset;
  int ZeilenPosi=Zeile*BUF_SPALTEN;
  for (int Spalte=0; Spalte<LCD_SPALTEN; Spalte++) {
    Lcd.write(ZeichenBuffer[ZeilenPosi+Spalte]);
  }
}
//=============================================================================

void ausgebenLcdZeileMonitor(int Zeile) {
  Zeile=Zeile+ZeilenOffset;
  int ZeilenPosi=Zeile*BUF_SPALTEN;
  for (int Spalte=0; Spalte<LCD_SPALTEN; Spalte++) {
    if (Zeile==ZeilenOffset && Spalte==LCD_SPALTEN-1) {
      if (ZeilenOffset>0) Serial.write(CODE_PFEIL_AUF); else Serial.write(ZeichenBuffer[ZeilenPosi+Spalte]);
    } else if (Zeile==ZeilenOffset+3 && Spalte==LCD_SPALTEN-1) {
      if (ZeilenOffset<MaxZeilenOffset) Serial.write(CODE_PFEIL_AB); else Serial.write(ZeichenBuffer[ZeilenPosi+Spalte]);
    } else {
      Serial.write(ZeichenBuffer[ZeilenPosi+Spalte]);
    }
  }
  Serial.println();
}
//=============================================================================

void ausgebenLcdZeilenMonitor() {
    for (int Zeile=0; Zeile<LCD_ZEILEN; Zeile++) {
      ausgebenLcdZeileMonitor(Zeile);
    }
    Serial.println(F("----------------"));
}
//=============================================================================

void ausgebenLcdZeilen() {
    for (int Zeile=0; Zeile<LCD_ZEILEN; Zeile++) {
      Lcd.setCursor(0, Zeile);
      ausgebenLcdZeile(Zeile);
    }
    if (ZeilenOffset>0) { Lcd.setCursor(LCD_SPALTEN-1, 0 ); Lcd.write(CODE_PFEIL_AUF); }
    if (ZeilenOffset<MaxZeilenOffset) { Lcd.setCursor(LCD_SPALTEN-1, LCD_ZEILEN-1 ); Lcd.write(CODE_PFEIL_AB); }
//    ausgebenLcdZeilenMonitor(); // LCD auf Seriellen Monitor ausgeben für Display-Bilder
}
//=============================================================================
//=============================================================================

void loeschenAbtastungen() {
  for (int Pin=0; Pin<ANZAHL_PINS; Pin++) {
    IstAbtastungen[Pin].UL=0;
    SollAbtastungen[Pin].UL=0;
    FehlAbtastungen[Pin].UL=0;
  }
}
//=============================================================================

void loeschenGleicheVerbindungen(union Abtastung Abtast[]) {
  for (int Pin1=0; Pin1<ANZAHL_PINS; Pin1++) {
    unsigned long Hilf=Abtast[Pin1].UL;
    if (Hilf!=0) {
      for (int Pin2=Pin1+1; Pin2<ANZAHL_PINS; Pin2++) {
        if (Hilf==Abtast[Pin2].UL) {
          Abtast[Pin2].UL=0;
        }
      }
    }
  }
}
//=============================================================================

void loeschenKnoten(union Abtastung Abtast[], const int Pin1, const int Pin2) {
  bitClear(Abtast[Pin1].UL, Pin2);
}
//=============================================================================

void loeschenVerbindung(union Abtastung Abtast[], const int Pin1, const int Pin2) {
    loeschenKnoten(Abtast, Pin1, Pin2);
    loeschenKnoten(Abtast, Pin2, Pin1);
}
//=============================================================================

void loeschenZeile(union Abtastung Abtast[], const int Pin) {
  Abtast[Pin].UL=0;
}
//=============================================================================

void loeschenSpalte(union Abtastung Abtast[], const int Pin2) {
  for (int Pin1=0; Pin1<ANZAHL_PINS; Pin1++) {
    bitClear(Abtast[Pin1].UL, Pin2);
  }
}
//=============================================================================

void loeschenEigenenKnoten(union Abtastung Abtast[]) {
  for (int Pin=0; Pin<ANZAHL_PINS; Pin++) {
    bitClear(Abtast[Pin].UL, Pin); // eigene Abtastung löschen
  }
}
//=============================================================================

int transferPin(const int Pin) {
  return pgm_read_byte(PinFolgenOffset+Pin);
}
//=============================================================================

void tauschenPinsVerbindung(unsigned long &Verbindung) {
  unsigned long Hilf=Verbindung;
  Verbindung=0;
  if (Hilf!=0) {
    for (int Pin=0; Pin<ANZAHL_PINS; Pin++)
      if (bitRead(Hilf, transferPin(Pin))) bitSet(Verbindung, Pin);
  }
}
//=============================================================================

void tauschenPinsAbtastungen(union Abtastung Abtast[]) {
  union Abtastung HilfAbtast[ANZAHL_PINS];
  for (int Pin=0; Pin<ANZAHL_PINS; Pin++) {
    HilfAbtast[Pin].UL=Abtast[Pin].UL;
    Abtast[Pin].UL=0;
  }
  for (int Pin1=0; Pin1<ANZAHL_PINS; Pin1++) {
    int TPin1=transferPin(Pin1);
    for (int Pin2=0; Pin2<ANZAHL_PINS; Pin2++) {
      int TPin2=transferPin(Pin2);
      if (bitRead(HilfAbtast[TPin1].UL, TPin2)) bitSet(Abtast[Pin1].UL, Pin2);
    }
  }
}
//=============================================================================

void einfuegenPinNamen(const int Pin) {
  int Offset=PinNamenOffset+PIN_MAMEN_LAENGE*Pin;
  for (int Zeichen=0; Zeichen<PIN_MAMEN_LAENGE; Zeichen++)
    einfuegenZeichen(pgm_read_byte(Offset+Zeichen));
}
//=============================================================================

int einfuegenPinNamenListe(const unsigned long Liste, const char Vorher, const char Nachher, const char Abschluss) {
  int Pins=0;
  for (int Pin=0; Pin<ANZAHL_PINS; Pin++) {
    if (bitRead(Liste, Pin)) {
      if (Vorher!=NULL_ZEICHEN) einfuegenZeichen(Vorher);
      einfuegenPinNamen(Pin);
      if (Nachher!=NULL_ZEICHEN) einfuegenZeichen(Nachher);
      Pins++;
    }
  }
  if (Abschluss!=NULL_ZEICHEN) einfuegenZeichen(Abschluss);
  return Pins;
}
//=============================================================================
//=============================================================================

int testenMasseschluss() {
  union Abtastung Masseschluss;
  boolean SondeMasseschluss;
  int I2CFehler=0;
  Masseschluss.UL=0;
  SondeMasseschluss=0;
  
  for (int Byt=0; Byt<ANZAHL_BYTES; Byt++) { // alle Pins auf 1 stetzen
    Wire.beginTransmission(PCF8574Adr[Byt]);
    Wire.write(0xFF);
    if (Wire.endTransmission()!=0) I2CFehler=I2CFehler|bit(Byt);
  }
  if (I2CFehler!=0) return I2CFehler;

  for (int Byt=0; Byt<ANZAHL_BYTES; Byt++) { // alle Pins negiert einlesen
    Wire.requestFrom(PCF8574Adr[Byt], 1); 
    while (Wire.available()<1) ;
    byte Abtast=~Wire.read();
    Masseschluss.B[Byt]=Abtast;
  }

  PinFehlerPins=Masseschluss.UL;
  SondeMasseschluss=digitalRead(SONDE_EINGANG_PIN)==LOW;

  if (Masseschluss.UL==0 && !SondeMasseschluss) return 0;

  tauschenPinsVerbindung(Masseschluss.UL);

  if (SondeMasseschluss) {
    loeschenBufferZeilen();
    einfuegenPGMText(Sonde_hat_Text); einfuegenNeueZeile(); 
    einfuegenPGMText(Masseschluss_Text);
  } else {
    einfuegenPGMText(Masseschluss2_Text); einfuegenNeueZeile();
    einfuegenPinNamenListe(Masseschluss.UL, NULL_ZEICHEN, LEER_ZEICHEN, NULL_ZEICHEN);
  }
  return MASSESCHLUSS_FEHLER;
}
//=============================================================================

int abtastenPins() {
  union Abtastung PruefMuster;

  int SystemFehler=0;
  SondeVerbindungen=0;
  PinFehlerPins=0; 
  AnzahlPinFehler=0;
  VerbindungenDa=false;
  loeschenAbtastungen();

  for (int Pin=0; Pin<ANZAHL_PINS; Pin++) { // Pins nacheinander auf L-Pegel legen  
    PruefMuster.UL=bit(Pin);
    for (int Byt=0; Byt<ANZAHL_BYTES; Byt++) {
        Wire.beginTransmission(PCF8574Adr[Byt]);
        Wire.write(~PruefMuster.B[Byt]);
        if (Wire.endTransmission()!=0) SystemFehler=SystemFehler|bit(Byt);
    }
    if (SystemFehler!=0) return SystemFehler; 

    delay(ABTAST_DELAY);

    for (int Byt=0; Byt<ANZAHL_BYTES; Byt++) { // Pegel der Pins einlesen
      Wire.requestFrom(PCF8574Adr[Byt], 1); 
      while (Wire.available()<1) ;
      byte Abtast=~Wire.read();
      IstAbtastungen[Pin].B[Byt]=Abtast;
    }
    if (IstAbtastungen[Pin].UL!=PruefMuster.UL)  VerbindungenDa=true;

    if ((IstAbtastungen[Pin].UL & PruefMuster.UL)==0) { // konnte Pin aktiviert werden?
      PinFehlerPins=PinFehlerPins | PruefMuster.UL; 
      AnzahlPinFehler++;
    }
    
    if (digitalRead(SONDE_EINGANG_PIN)==LOW) bitSet(SondeVerbindungen, Pin);
  }

  for (int Byt=0; Byt<ANZAHL_BYTES; Byt++) { // alle Pins wieder auf H-Pegel
    Wire.beginTransmission(PCF8574Adr[Byt]);
    Wire.write(0xFF);
    if (Wire.endTransmission()!=0) SystemFehler=SystemFehler|bit(Byt);
  }

  if (SystemFehler==0 && AnzahlPinFehler>0) SystemFehler=PIN_FEHLER;
  return SystemFehler;
}

//============================================================================
//============================================================================

void speichernAbtastungen() {
  EEPROM.put(ABTASTUNGEN_EEPROM_ADRESSE, IstAbtastungen);
  EEPROM.put(GELERNT_CRC_EEPROM_ADRESSE, AbtastungenCrc);
  piezoPuls(SPEICHERN_TON_DAUER); 
  ZeigeGespeichertTimer=ZEIGE_GESPEICHERT_DAUER;
  ZeigeLcdKontrastTimer=0;
}
//=============================================================================

void bearbeitenSonde() {
  tauschenPinsVerbindung(SondeVerbindungen);

  einfuegenPGMText(Sonde_Text);
  einfuegenNeueZeile();
  
  int AnzahlSondeVerbindungen=0;

  for (int Pin=0; Pin<ANZAHL_PINS; Pin++) {
    if (bitRead(SondeVerbindungen, Pin)) {
      einfuegenPinNamen(Pin); 
      einfuegenZeichen(LEER_ZEICHEN);
      AnzahlSondeVerbindungen++;
    }
  }
  nachtragenZahlPosi(AnzahlSondeVerbindungen, ANZAHL_VERBIN_POS); // Anzahl der Sondenverbindungen nachtragen
}
//=============================================================================

void bearbeitenVerbindungen() {

  tauschenPinsAbtastungen(IstAbtastungen);
  loeschenGleicheVerbindungen(IstAbtastungen);
  loeschenEigenenKnoten(IstAbtastungen);

  einfuegenHexWord(AbtastungenCrc); 
  einfuegenPGMText(Verb_Text); 
  einfuegenNeueZeile();
  
  int AnzahlVerbindungen=0;
  for (int Pin1=0; Pin1<ANZAHL_PINS; Pin1++) {
    unsigned long Hilf=IstAbtastungen[Pin1].UL;
    if (Hilf!=0) {
      int Verbindungen=zaehlenEinsen(Hilf);
      if (BufferSpalte+(Verbindungen+1)*PIN_MAMEN_LAENGE>=BUF_SPALTEN) einfuegenNeueZeile();
      einfuegenPinNamen(Pin1);
      einfuegenPinNamenListe(Hilf, VERBINDUNG_TRENNER, NULL_ZEICHEN, VERBINDUNGS_ABSCHLUSS);
      if (Verbindungen>1) einfuegenNeueZeile();
      AnzahlVerbindungen++;
    }
  }
  nachtragenZahlPosi(AnzahlVerbindungen, ANZAHL_VERBIN_POS); // Anzahl der Verbindungen nachtragen
}
//=============================================================================

void bearbeitenFehlerPinsVertauscht() {
  boolean WarOder=false;
  for (int Pin1=0; Pin1<ANZAHL_PINS; Pin1++) {
    unsigned long Hilf1=FehlAbtastungen[Pin1].UL;
    if (Hilf1!=0) {
      for (int Pin2=Pin1+1; Pin2<ANZAHL_PINS; Pin2++) {
        unsigned long Hilf2=FehlAbtastungen[Pin2].UL;
        if (Hilf1==Hilf2) {  // gleiche Fehler an Pin1 und Pin2
          if (Hilf1==(SollAbtastungen[Pin1].UL^SollAbtastungen[Pin2].UL)) {
            einfuegenPinNamen(Pin1);
            einfuegenZeichen(PIN_GETAUSCHT_TRENNER);
            einfuegenPinNamen(Pin2);

            if (zaehlenEinsen(Hilf1)==2) { // 2 Fehler an Pin1?
              int Pin3=suchenEins(Hilf1, 0);
              int Pin4=suchenEins(Hilf1, Pin3+1);
              Hilf1=FehlAbtastungen[Pin3].UL;
              Hilf2=FehlAbtastungen[Pin4].UL;
              if (Hilf1==Hilf2) {  // gleiche Fehler an Pin3 und Pin4
                if (Hilf1==(SollAbtastungen[Pin3].UL^SollAbtastungen[Pin4].UL)) {
                  WarOder=true;
                  einfuegenPGMText(Oder_Text);
                  einfuegenNeueZeile();
                  einfuegenPinNamen(Pin3);
                  einfuegenZeichen(PIN_GETAUSCHT_TRENNER);
                  einfuegenPinNamen(Pin4);
                  loeschenZeile(FehlAbtastungen, Pin3);
                  loeschenZeile(FehlAbtastungen, Pin4);
                  loeschenSpalte(FehlAbtastungen, Pin3);
                  loeschenSpalte(FehlAbtastungen, Pin4);
                }
              }
            }

            einfuegenPGMText(Getauscht_Text);
            einfuegenNeueZeile();
            if (WarOder) Trennung=true;
            loeschenZeile(FehlAbtastungen, Pin1);
            loeschenZeile(FehlAbtastungen, Pin2);
            loeschenSpalte(FehlAbtastungen, Pin1);
            loeschenSpalte(FehlAbtastungen, Pin2);
            AnzahlFehler++;
          }
        }
      }
    }
  }
}
//=============================================================================

void bearbeitenFehlerPinFehltZuviel() {

  boolean ZuLoeschen[ANZAHL_PINS];
  for (int Pin=0; Pin<ANZAHL_PINS; Pin++) ZuLoeschen[Pin]=false;

  for (int Pin=0; Pin<ANZAHL_PINS; Pin++) {
    unsigned long Hilf1=FehlAbtastungen[Pin].UL;
    if (Hilf1!=0 && zaehlenEinsen(Hilf1)>=ANZAHL_KNOTEN_FEHLT_ZUVIEL) {
      if (Hilf1==IstAbtastungen[Pin].UL) {
        ZuLoeschen[Pin]=true;
        einfuegenPGMText(Pin_Zuviel_Text);           
        einfuegenPinNamen(Pin);
        einfuegenNeueZeile();
        einfuegenPinNamen(Pin);
        einfuegenPinNamenListe(Hilf1, PIN_ZUVIEL_TRENNER, NULL_ZEICHEN, VERBINDUNGS_ABSCHLUSS);
        einfuegenNeueZeile();
        Trennung=true;
        AnzahlFehler++;
      }
      if (Hilf1==SollAbtastungen[Pin].UL) {
        ZuLoeschen[Pin]=true;
        einfuegenPGMText(Pin_Fehlt_Text);           
        einfuegenPinNamen(Pin);
        einfuegenNeueZeile();
        einfuegenPinNamen(Pin);
        einfuegenPinNamenListe(Hilf1, PIN_FEHLT_TRENNER, NULL_ZEICHEN, VERBINDUNGS_ABSCHLUSS);
        einfuegenNeueZeile();
        Trennung=true;
        AnzahlFehler++;
      }
    }
  }

  for (int Pin1=0; Pin1<ANZAHL_PINS; Pin1++) {
    if (ZuLoeschen[Pin1]) {
      unsigned long Hilf1=FehlAbtastungen[Pin1].UL;
      for (int Pin2=0; Pin2<ANZAHL_PINS; Pin2++) {
        if (bitRead(Hilf1, Pin2)) {
          loeschenVerbindung(IstAbtastungen, Pin1, Pin2);
          loeschenVerbindung(SollAbtastungen, Pin1, Pin2);
          loeschenVerbindung(FehlAbtastungen, Pin1, Pin2);
        }
      }
    }
  }
}
//=============================================================================

void bearbeitenFehlVerbindungen() {
  EEPROM.get(ABTASTUNGEN_EEPROM_ADRESSE, SollAbtastungen);

  tauschenPinsAbtastungen(IstAbtastungen);
  tauschenPinsAbtastungen(SollAbtastungen);

  loeschenEigenenKnoten(IstAbtastungen);
  loeschenEigenenKnoten(SollAbtastungen);

  for (int Pin=0; Pin<ANZAHL_PINS; Pin++) {
    FehlAbtastungen[Pin].UL=IstAbtastungen[Pin].UL^SollAbtastungen[Pin].UL;
  }

  einfuegenHexWord(AbtastungenCrc); 
  einfuegenPGMText(Fehler_Text); 
  einfuegenNeueZeile();

  AnzahlFehler=0;
  int AltAnzahlFehler;
  do {
    AltAnzahlFehler=AnzahlFehler;
    bearbeitenFehlerPinsVertauscht();
    bearbeitenFehlerPinFehltZuviel();
  } while (AltAnzahlFehler!=AnzahlFehler);

// alle anderen Fehler
  for (int Pin1=0; Pin1<ANZAHL_PINS; Pin1++) {
    unsigned long Hilf=FehlAbtastungen[Pin1].UL;
    if (Hilf!=0) { // Fehler an Pin1?
      for (int Pin2=Pin1+1; Pin2<ANZAHL_PINS; Pin2++) {
        if (bitRead(Hilf, Pin2)) {
          einfuegenPinNamen(Pin1);
          einfuegenZeichen(VERBINDUNG_TRENNER);
          einfuegenPinNamen(Pin2);
          if (bitRead(IstAbtastungen[Pin1].UL, Pin2)) {
            einfuegenPGMText(Zuviel_Text);           
          } else {
            einfuegenPGMText(Fehlt_Text);           
          }
          einfuegenNeueZeile();
          AnzahlFehler++;
        }
      }
    }
  }
  nachtragenZahlPosi(AnzahlFehler, ANZAHL_FEHLER_POS); // Anzahl der Fehler nachtragen
}
//=============================================================================

void setzeLcdKontrast() {
  LcdKontrast=constrain(LcdKontrast, MIN_LCD_KONTRAST, MAX_LCD_KONTRAST);
  EEPROM.update(LCD_KONTRAST_EEPROM_ADRESSE, LcdKontrast); 
  analogWrite(LCD_KONTRAST_AUSGANG_PIN, map(LcdKontrast, MIN_LCD_KONTRAST, MAX_LCD_KONTRAST, 0, 255)); 
}
//=============================================================================

void setzeAnzeigeModus() {
  if (AnzeigeModus>=ANZAHL_ANZEIGE_MODUS) AnzeigeModus=0;
  EEPROM.update(ANZEIGEMODUS_EEPROM_ADRESSE, AnzeigeModus);
  PinNamenOffset=pgm_read_word(&(PinNamen[AnzeigeModus]));
  PinFolgenOffset=pgm_read_word(&(PinFolgen[AnzeigeModus]));
}
//=============================================================================

void zeigenTitel() {
  loeschenBufferZeilen();
  for (int Zeile=0; Zeile<4; Zeile++) einfuegenPGMText(Titel_Texte[Zeile]);
  ausgebenLcdZeilen();
  delay(TITEL_ANZEIGE_DAUER);
  loeschenBufferZeilen();
}
//=============================================================================

void scannePCF(bool Anzeigen) {
  for (int Adr=PCF8574_SLAVE_ADR; Adr<PCF8574_SLAVE_ADR+ANZAHL_BYTES; Adr++) {
    Wire.begin();
    Wire.beginTransmission(Adr);
    byte BausteinDa=Wire.endTransmission();
    Wire.end();
    if (BausteinDa==0) {
      PCF8574Adr[Adr&3]=Adr;
    }
  }
  for (int Adr=PCF8574A_SLAVE_ADR; Adr<PCF8574A_SLAVE_ADR+ANZAHL_BYTES; Adr++) {
    Wire.begin();
    Wire.beginTransmission(Adr);
    byte BausteinDa=Wire.endTransmission();
    Wire.end();
    if (BausteinDa==0) {
      PCF8574Adr[Adr&3]=Adr;
    }
  }
  int Gefunden=0;
  for (int Baustein=0; Baustein<4; Baustein++) {
    einfuegenHexNibble(Baustein);
    einfuegenZeichen(':');
    einfuegenZeichen(' ');
    int Adr=PCF8574Adr[Baustein];
    if (Adr>=PCF8574_SLAVE_ADR && Adr<=PCF8574_SLAVE_ADR+3) {
        einfuegenPGMText(PCF8574_Text);
        Gefunden++;
    } else {
      if (Adr>=PCF8574A_SLAVE_ADR && Adr<=PCF8574A_SLAVE_ADR+3) {
          einfuegenPGMText(PCF8574A_Text); 
          Gefunden++;
      } else {
        einfuegenPGMText(PCF8574fehlt_Text); 
      }
    }
    einfuegenNeueZeile();
  }
  if (Anzeigen) {
    ausgebenLcdZeilen();
    delay(TITEL_ANZEIGE_DAUER);
    while(Gefunden!=4) { ; }
  }
  Wire.begin();
}
//=============================================================================

void zeigeASCII() {
  for (int i=128; i<256; i++) {
    Serial.print(i,OCT);
    Serial.print("=0x");
    Serial.print(i,HEX);
    Serial.print(" ");
    Serial.print(char(i));
    Serial.print('\t');
    if (i%8==7) Serial.println();
  }
  while(1);
}
//=============================================================================

void LcdInit() {
  Lcd.begin(LCD_SPALTEN, LCD_ZEILEN);
  Lcd.createChar(CODE_PFEIL_AUF, PfeilAuf);
  Lcd.createChar(CODE_PFEIL_AB, PfeilAb);
}

//=============================================================================

void setup() {
  LcdInit();
  Serial.begin(9600);
  Serial.println(F("EGS Abschlusspruefung Teil 1, Fruehjahr 2025 ")); 
//  zeigeASCII(); // welche Codes haben ä. ö, ü, usw. ?

  pinMode(SONDE_EINGANG_PIN, INPUT_PULLUP);
  pinMode(BETRIEBSMODUS_EINGANG_PIN, INPUT_PULLUP);
  pinMode(PIEZO_AUSGANG_PIN, OUTPUT);
  digitalWrite(PIEZO_AUSGANG_PIN, LOW);

  loeschenBufferZeilen();
  loeschenAbtastungen();
  SondeVerbindungen=0;
  VerbindungenDa=false;
  AnzahlFehler=0;
  PinFehlerPins=0;
  AnzahlPinFehler=0;

  ZeichenBufferCrc=CRC_START;
  AltZeichenBufferCrc=CRC_START;
  AbtastungenCrc=CRC_START;

  ZeilenOffset=0;
  ZeigeLcdKontrastTimer=0;
  ZeigeGespeichertTimer=0;
  DirektAnzeigen=false;

  AnzeigeModus=EEPROM.read(ANZEIGEMODUS_EEPROM_ADRESSE);
  setzeAnzeigeModus();

  LcdKontrast=EEPROM.read(LCD_KONTRAST_EEPROM_ADRESSE);
  setzeLcdKontrast();
  
  AnzeigeModusTaste=TASTE_NULL;
  LcdKontrastTaste=TASTE_NULL;
  BetriebsModus=digitalRead(BETRIEBSMODUS_EINGANG_PIN); // L=Sonde, H=Verbindungen

  scannePCF(false);
  zeigenTitel();
}
//=============================================================================

void zeigenI2CFehler(unsigned long Fehler) {
  loeschenBufferZeilen();
  einfuegenPGMText(I2C_Fehler_Text); 
  einfuegenNeueZeile();
  for (int Byt=0; Byt<ANZAHL_BYTES; Byt++) {
    if (bitRead(Fehler, Byt)) {
      einfuegenZeichen('1'+Byt);
      einfuegenZeichen(LEER_ZEICHEN);
    }
  }
}
//=============================================================================

void zeigenPinFehler() {
  einfuegenZahl(AnzahlPinFehler, LEER_ZEICHEN);
  einfuegenPGMText(Pin_Fehler_Text); 
  einfuegenNeueZeile();
  tauschenPinsVerbindung(PinFehlerPins);
  einfuegenPinNamenListe(PinFehlerPins, NULL_ZEICHEN, LEER_ZEICHEN, NULL_ZEICHEN);
}
//=============================================================================

void loop() {
  loeschenBufferZeilen();
  DirektAnzeigen=false;

//-----------------------------------------------------------------------------
// Eingabe
//-----------------------------------------------------------------------------
  int ScrollTaste=TASTE_NULL;
  int ScrollenSpannung=analogRead(SCROLLEN_SPANNUNG_PIN);
  if (ScrollenSpannung<UMIN) ScrollTaste=TASTE_GND;
  if (ScrollenSpannung>UMAX) ScrollTaste=TASTE_VCC;

  int AltAnzeigeModusTaste=AnzeigeModusTaste;
  AnzeigeModusTaste=TASTE_NULL;
  int AnzeigeModusSpannung=analogRead(ANZEIGE_MODUS_SPANNUNG_PIN);
  if (AnzeigeModusSpannung<UMIN) AnzeigeModusTaste=TASTE_GND;
  if (AnzeigeModusSpannung>UMAX) AnzeigeModusTaste=TASTE_VCC;

  int AltLcdKontrastTaste=LcdKontrastTaste;
  LcdKontrastTaste=TASTE_NULL;
  int LcdKontrastSpannung=analogRead(LCD_KONTRAST_SPANNUNG_PIN);
  if (LcdKontrastSpannung<UMIN) LcdKontrastTaste=TASTE_GND;
  if (LcdKontrastSpannung>UMAX) LcdKontrastTaste=TASTE_VCC;

  boolean AltSpeichernModus=SpeichernModus;
  SpeichernModus=false; VergleichenModus=false;  
  int SpeichernVergleichenSpannung=analogRead(SPEICHERN_VERGLEICHEN_SPANNUNG_PIN);
  if (SpeichernVergleichenSpannung<UMIN) { VergleichenModus=true; ZeigeGespeichertTimer=0; } // GND
  if (SpeichernVergleichenSpannung>UMAX) SpeichernModus=true; // VCC

  BetriebsModus=digitalRead(BETRIEBSMODUS_EINGANG_PIN);

//-----------------------------------------------------------------------------
// Verarbeitung
//-----------------------------------------------------------------------------
  int AltZeilenOffset=ZeilenOffset;
  if (ScrollTaste==TASTE_GND) { ZeilenOffset--; } // aufwärts scrollen
  if (ScrollTaste==TASTE_VCC) { ZeilenOffset++; } // abwärts scrollen
  ZeilenOffset=constrain(ZeilenOffset, 0, MaxZeilenOffset);
  if (ScrollTaste!=TASTE_NULL && AltZeilenOffset!=ZeilenOffset) DirektAnzeigen=true;

  int AltAnzeigeModus=AnzeigeModus;
  if (AltAnzeigeModusTaste==TASTE_NULL && AnzeigeModusTaste==TASTE_GND) { if (AnzeigeModus>0) AnzeigeModus--; else AnzeigeModus=ANZAHL_ANZEIGE_MODUS-1; } // ... D, C, B, A
  if (AltAnzeigeModusTaste==TASTE_NULL && AnzeigeModusTaste==TASTE_VCC) { if (AnzeigeModus<ANZAHL_ANZEIGE_MODUS-1) AnzeigeModus++; else AnzeigeModus=0; } // A, B, C, D, ...
  if (AltAnzeigeModus!=AnzeigeModus) setzeAnzeigeModus();

  int AltLcdKontrast=LcdKontrast;
  if (ZeigeLcdKontrastTimer>0 && AltLcdKontrastTaste==TASTE_NULL && LcdKontrastTaste==TASTE_GND && LcdKontrast<MAX_LCD_KONTRAST) LcdKontrast++;
  if (ZeigeLcdKontrastTimer>0 && AltLcdKontrastTaste==TASTE_NULL && LcdKontrastTaste==TASTE_VCC && LcdKontrast>MIN_LCD_KONTRAST) LcdKontrast--; 
  if (LcdKontrastTaste!=TASTE_NULL) { ZeigeLcdKontrastTimer=ZEIGE_LCD_KONTRAST_DAUER; ZeigeGespeichertTimer=0; }
  if (AltLcdKontrast!=LcdKontrast) setzeLcdKontrast();
  
//-----------------------------------------------------------------------------
// Ausgabe vorbereiten
//-----------------------------------------------------------------------------
  if (ZeigeLcdKontrastTimer>0) {
    ZeigeLcdKontrastTimer--;
    einfuegenPGMText(Lcd_Kontrast_Text);
    einfuegenZahl(LcdKontrast, NULL_ZEICHEN);
    DirektAnzeigen=true;
  } else if (ZeigeGespeichertTimer>0) {
    ZeigeGespeichertTimer--;
    einfuegenPGMText(Verbindungen_Text);
    einfuegenNeueZeile();
    einfuegenPGMText(Gespeichert_Text);
    DirektAnzeigen=true;
  } else {
    einfuegenZeichen('A'+AnzeigeModus); einfuegenZeichen(ANZEIGE_MODUS_TRENNER);
    int Fehler=testenMasseschluss();
    if (Fehler==0) {
      Fehler=abtastenPins();
      if (Fehler==0) {
        AbtastungenCrc=rechnenCrc((byte*)IstAbtastungen, sizeof(IstAbtastungen));
        if (!BetriebsModus) { // L=Sonde
          bearbeitenSonde();
         } else { // H=Verbindungen
          if (!VergleichenModus) {
            if (!AltSpeichernModus && SpeichernModus) speichernAbtastungen();
            bearbeitenVerbindungen(); 
          } else {
            bearbeitenFehlVerbindungen();
          }
        }
      }
    }
    if (Fehler>0) {
      zeigenI2CFehler(Fehler);
      DirektAnzeigen=true;
    } else if (Fehler==PIN_FEHLER){
      zeigenPinFehler();
      DirektAnzeigen=true;
    }
  }

//-----------------------------------------------------------------------------
// Ausgabe
//-----------------------------------------------------------------------------
  word LetzterZeichenBufferCrc=ZeichenBufferCrc;
  ZeichenBufferCrc=rechnenCrc((byte*)ZeichenBuffer, sizeof(ZeichenBuffer));
  
  if (LetzterZeichenBufferCrc==ZeichenBufferCrc || DirektAnzeigen) { // 2x gleiche Anzeige?
    if (AltZeichenBufferCrc!=ZeichenBufferCrc || DirektAnzeigen) {  // hat sich die Anzeige geändert?
      if (AltZeichenBufferCrc!=ZeichenBufferCrc && !DirektAnzeigen) { // neue Verbindungsliste
        LcdInit();
        ZeilenOffset=0; 
        piezoPuls(ANZEIGE_TON_DAUER);
      }
      ausgebenLcdZeilen();
      AltZeichenBufferCrc=ZeichenBufferCrc;
    }
  }
  
  while (millis()%LOOP_DAUER!=0); 
}
//-----------------------------------------------------------------------------
// Programmende
//-----------------------------------------------------------------------------
